#include "MTWindow.h"
#include "MTApplication.h"
#include "MTUtilities.h"
#include "IErrors.h"
#include "MTExceptions.h"
#include "MTFileList.h"

//	MTWindow::MTWindow()
//	
//	constructor
MTWindow::MTWindow()
:theWindow(nil), horzScroll(nil), vertScroll(nil), owner(nil)
{
	scrollActionProcPtr = nil;
	rootControl = nil;
}

//	MTWindow::~MTWindow()
//	
//	destructor, closes window
MTWindow::~MTWindow()
{
	if(scrollActionProcPtr)
	{
		DisposeControlActionUPP(scrollActionProcPtr);
		
		scrollActionProcPtr = nil;
	}
	
	Close();
}

//	void MTWindow::Open(void)
//	
//	opens window, dispatches open event
void MTWindow::Open(void)
{
	Rect						where, standardSizeRect;
	Str255						name;
	MTWindowExtraWindowInfoRec	* record = (MTWindowExtraWindowInfoRec *)NewPtr(sizeof(MTWindowExtraWindowInfoRec));
	OSErr						theErr;
	
	GetWindowSize(&where);
	GetWindowName(name);
	
	record->data = this;
	
	theWindow = NewCWindow((CWindowRecord *)record, &where, name, 0, GetWindowType(), (WindowPtr)-1, 1, nil);
	if(!theWindow)
		throw MTException("Error creating window");
	
	theErr = SetThemeWindowBackground(theWindow, kThemeBrushDialogBackgroundActive, 1);
	if(theErr)
		throw MTOSException(theErr, "Error setting window background");
	
	SetWRefCon(theWindow, kMTWindowTypeFlag);						// <- this constant is used
	RepositionWindow(theWindow, nil, kWindowCascadeOnMainScreen);	// to mark a window as
																	// a window which also
																	// has an MTWindow record
	GetWindowStandardSize(&standardSizeRect);
	SetWindowStandardState(theWindow, &standardSizeRect);
	
	theErr = CreateRootControl(theWindow, &rootControl);
	if(theErr)
		throw MTOSException(theErr, "Error creating root control");
	
	HandleOpenEvent();
	
	ShowWindow(theWindow);	// the window was created closed for cleaner drawing purposes, show it
	
	InvalWindowRect(theWindow, &standardSizeRect);
	DrawControls(theWindow);
	
	AdvanceKeyboardFocus(theWindow);
	
	HandleUpdateEvent();
}

//	void MTWindow::Close(void)
//	
//	closes window, dispatches close event
void MTWindow::Close(void)
{
	if(theWindow)
	{
		HandleCloseEvent();
		
		CloseWindow(theWindow);
		
		DisposePtr((Ptr)theWindow);
		theWindow = nil;
		
		gTheApp->CloseAllOwnedBy(this);
	}
}

//	UInt8 MTWindow::operator==(WindowPtr rhs)
//	
//	compares an MTWindow to 'this'
//	
//	rhs = window to test
//	returns if the windows are equal
UInt8 MTWindow::operator==(WindowPtr rhs)
{
	if(GetWRefCon(rhs) == kMTWindowTypeFlag)
	{
		MTWindowExtraWindowInfoRec	* data = (MTWindowExtraWindowInfoRec *)rhs;	// cast window ptr to extended info record
		
		return this == data->data;	// extract MTWindow
	}
	
	return 0;
}

//	UInt8 MTWindow::AutoHandleControls(EventRecord * theEvent)
//	
//	does automatic handling of controls
//	
//	theEvent = pointer to mouse down event in window content region
//	returns if the event was unhandled
UInt8 MTWindow::AutoHandleControls(EventRecord * theEvent)
{
	Point					where;
	ControlRef				whichControl;
	SInt16					whichPart;
	MTWindowControlType		type;
	MTGrafPortSaver			saver(theWindow);
	
	where = theEvent->where;
	GlobalToLocal(&where);
	
	whichControl = FindControlUnderMouse(where, theWindow, &whichPart);
	
	if(whichControl)
	{
		type = GetControlType(whichControl);	// check control's refcon to see what type of control it is
		
		switch(type)
		{
			case kHorzScrollBarType:
			case kVertScrollBarType:
			{
				whichPart = HandleControlClick(whichControl, where, theEvent->modifiers, (RoutineDescriptor *)-1);	// use control's tracking proc
			}
			break;
			
			case kGenericControlType:
			case kButtonControlType:
			default:
				whichPart = HandleControlClick(whichControl, where, theEvent->modifiers, nil);	// use standard tracking proc
				break;
		}
		
		switch(type)	// dispatch event
		{
			case kGenericControlType:
				HandleControlEvent(whichControl, whichPart);
				break;
			
			case kButtonControlType:
				if(whichPart)
				{
					HandleButtonEvent(whichControl);
				}
				break;
			
			case kHorzScrollBarType:
			case kVertScrollBarType:
			default:
				break;
		}
		
		return 0;
	}
	
	return 1;
}

//	pascal void MTWindow::ScrollActionProc(ControlRef theControl, ControlPartCode partCode)
//	
//	handles scroll bar movement
//	
//	theControl = control to handle
//	partCode = part clicked in
pascal void MTWindow::ScrollActionProc(ControlRef theControl, ControlPartCode partCode)
{
	SInt32		currentValue =	GetControl32BitValue(theControl);
	SInt32		viewSize =		GetControlViewSize(theControl);
	SInt32		delta = 0;
	MTWindow	* theWindow = GetMTWindowFromWindowPtr(GetControlOwner(theControl));
	
	switch(partCode)
	{
		case kControlUpButtonPart:
			if(currentValue > GetControl32BitMinimum(theControl))
				delta = -1;
			
			break;
		
		case kControlDownButtonPart:
			if(currentValue < GetControl32BitMaximum(theControl))
				delta = 1;
			
			break;
		
		case kControlPageUpPart:	// page by view size
			if(currentValue > GetControl32BitMinimum(theControl))
				delta = -viewSize;
			
			break;
		
		case kControlPageDownPart:	// page by view size
			if(currentValue < GetControl32BitMaximum(theControl))
				delta = viewSize;
			
			break;
	}
	
	if(delta)
		SetControl32BitValue(theControl, currentValue + delta);	// apply delta
	
	if((partCode != kControlIndicatorPart) && delta == 0)	// we're done if there was no delta
		return;
	
	if(theWindow)	// dispatch event if we have a window
	{
		if(theWindow->IsHorzScroll(theControl))
		{
			theWindow->HandleScrollEvent(kHorzScrollBarType, currentValue + delta);
		}
		else
		{
			if(theWindow->IsVertScroll(theControl))
			{
				theWindow->HandleScrollEvent(kVertScrollBarType, currentValue + delta);
			}
		}
	}
}

MTFileList * MTWindow::GetOwningFileList(void)
{
	MTFileList	* fileList;
	MTWindow	* traverse = this;
	
	while(traverse)
	{
		fileList = dynamic_cast<MTFileList *>(traverse);
		
		if(fileList)
			return fileList;
		
		traverse = traverse->GetOwner();
	}
	
	return nil;
}

//	void MTWindow::RegisterHorzScrollControl(ControlHandle theControl)
//	
//	marks a control as a horizontal scroll bar
//	
//	theControl = target control
void MTWindow::RegisterHorzScrollControl(ControlHandle theControl)
{
	if(!scrollActionProcPtr)	// get UPP to scroll proc
	{
		scrollActionProcPtr = NewControlActionUPP(ScrollActionProc);
	}
	
	if(!scrollActionProcPtr)
	{
		DoFatalError("Error creating scroll bar proc ptr");
		
		ExitToShell();
	}
	
	SetControlReference(theControl, kHorzScrollBarType);	// register control type
	SetControlAction(theControl, scrollActionProcPtr);		// set action proc
}

//	void MTWindow::RegisterVertScrollControl(ControlHandle theControl)
//	
//	marks a control as a vertical scroll bar
//	
//	theControl = target control
void MTWindow::RegisterVertScrollControl(ControlHandle theControl)
{
	if(!scrollActionProcPtr)	// get UPP to scroll proc
	{
		scrollActionProcPtr = NewControlActionUPP(ScrollActionProc);
	}
	
	if(!scrollActionProcPtr)
	{
		DoFatalError("Error creating scroll bar proc ptr");
		
		ExitToShell();
	}
	
	SetControlReference(theControl, kVertScrollBarType);	// register control type
	SetControlAction(theControl, scrollActionProcPtr);		// set action proc
}

//	void MTWindow::CreateScrollBars(UInt8 hasHorz, UInt8 hasVert)
//	
//	creates a window's scroll bars
//	
//	hasHorz = has horizontal scroll bar
//	hasVert = has vertical scroll bar
void MTWindow::CreateScrollBars(UInt8 hasHorz, UInt8 hasVert)
{
	UInt32	width = theWindow->portRect.right - theWindow->portRect.left;
	UInt32	height = theWindow->portRect.bottom - theWindow->portRect.top;
	Rect	where, padAmount;
	
	GetWindowContentPadding(&padAmount);
	
	if(hasHorz)
	{
		where.left =	-1							+ padAmount.left;
		where.top =		height - kScrollBarSize + 1	- padAmount.bottom;
		where.right =	width - kScrollBarSize + 2	- padAmount.right;
		where.bottom =	height + 1					- padAmount.bottom;
		
		horzScroll = NewControl(theWindow, &where, "\p", 1, 0, 0, 0, kControlScrollBarLiveProc, 0);
		
		RegisterHorzScrollControl(horzScroll);
	}
	else
	{
		horzScroll = nil;
	}
	
	if(hasVert)
	{
		where.left =	width - kScrollBarSize + 1		- padAmount.right;
		where.top =		-1								+ padAmount.top;
		where.right =	width + 1						- padAmount.right;
		where.bottom =	height - kScrollBarSize + 2		- padAmount.bottom;
		
		vertScroll = NewControl(theWindow, &where, "\p", 1, 0, 0, 0, kControlScrollBarLiveProc, 0);
		
		RegisterVertScrollControl(vertScroll);
	}
	else
	{
		vertScroll = nil;
	}
}

void MTWindow::MoveScrollBars(UInt32 newWidth, UInt32 newHeight)
{
	Rect	padAmount, where;
	
	GetWindowContentPadding(&padAmount);
	
	if(horzScroll)
	{
		where.left =	-1								+ padAmount.left;
		where.top =		newHeight - kScrollBarSize + 1	- padAmount.bottom;
		where.right =	newWidth - kScrollBarSize + 2	- padAmount.right;
		where.bottom =	newHeight + 1					- padAmount.bottom;
		
		SetControlBounds(horzScroll, &where);
		DrawOneControl(horzScroll);
	}
	
	if(vertScroll)
	{
		where.left =	newWidth - kScrollBarSize + 1	- padAmount.right;
		where.top =		-1								+ padAmount.top;
		where.right =	newWidth + 1					- padAmount.right;
		where.bottom =	newHeight - kScrollBarSize + 2	- padAmount.bottom;
		
		SetControlBounds(vertScroll, &where);
		DrawOneControl(vertScroll);
	}
}

//	MTWindowControlType MTWindow::GetControlType(ControlHandle theControl)
//	
//	gets a control's type
//	
//	theControl = target control
//	returns control's type
MTWindowControlType MTWindow::GetControlType(ControlHandle theControl)
{
	return (MTWindowControlType)GetControlReference(theControl);
}

//	MTWindow * GetMTWindowFromWindowPtr(WindowPtr theWindow)
//	
//	looks up an MTWindow from a WindowPtr
//	
//	theWindow = target window
//	returns MTWindow associated with WindowPtr
MTWindow * GetMTWindowFromWindowPtr(WindowPtr theWindow)
{
	if(theWindow == nil)
		return nil;
	
	if(GetWRefCon(theWindow) == kMTWindowTypeFlag)	// is this an extended window record?
		return ((MTWindowExtraWindowInfoRec *)theWindow)->data;	// it is, look up MTWindow
	
	return nil;	// it isn't return nil
}